#version 120

//Shader by LoLip_p

#include "/settings.glsl"

in vec2 TexCoords;

uniform float viewWidth, viewHeight;
uniform sampler2D colortex0;
uniform float frameTimeCounter;
uniform float isSneaking;

vec2 cameraShake(float frameTimeCounter, float intensity) {
    return vec2(
        intensity * 0, // Horizontal shake
        intensity * cos(frameTimeCounter * 1.85)  // Vertical shake
    );
}

void main() {
    vec3 color;

    // Apply zoom
    vec2 centeredCoord = (TexCoords - 0.5) * ZOOM + 0.5;
    float distance = length(centeredCoord - 0.5);

    // Camera shake effect
	vec2 shake = cameraShake(frameTimeCounter, INTENSITY_CAM_SHAKE);

    // Fisheye effect
    vec2 fisheyeUV = (centeredCoord + shake - 0.5) * (1.0 + DIST_STRENGTH * distance * distance) + 0.5;

	// Chromatic aberration
    vec2 redUV = fisheyeUV + vec2(CHROMATIC_ABERRATION, CHROMATIC_ABERRATION) * distance;
    vec2 blueUV = fisheyeUV + vec2(-CHROMATIC_ABERRATION, -CHROMATIC_ABERRATION) * distance;
    color.r = texture2D(colortex0, redUV).r;
    color.g = texture2D(colortex0, fisheyeUV).g;
    color.b = texture2D(colortex0, blueUV).b;

    // Sharpening
    vec2 texelSize = 1.0 / textureSize(colortex0, 0);
    vec3 sharpenedColor = vec3(0.0);

    // Define the sharpening kernel
    sharpenedColor += texture2D(colortex0, fisheyeUV + texelSize * vec2(-1.0,  0.0)).rgb * -1.0;
    sharpenedColor += texture2D(colortex0, fisheyeUV + texelSize * vec2( 1.0,  0.0)).rgb * -1.0;
    sharpenedColor += texture2D(colortex0, fisheyeUV + texelSize * vec2( 0.0, -1.0)).rgb * -1.0;
    sharpenedColor += texture2D(colortex0, fisheyeUV + texelSize * vec2( 0.0,  1.0)).rgb * -1.0;
    sharpenedColor += texture2D(colortex0, fisheyeUV).rgb * 5.0;

    // Combine the sharpened color with the original color
    color = mix(color, sharpenedColor, SHARPNESS_STRENGTH);
	color = clamp(color, 0, 1);

	color = pow(color, vec3(0.45f));

    #if SCANLINE == 1
	float scanline = sin(fisheyeUV.y * SCANLINE_WIDTH * 1.5) * SCANLINE_STRENGTH;
    color.rgb += scanline;
    #endif

    #if GRAIN == 1
    float noise = (fract(sin(dot(fisheyeUV, vec2(12.9898, 78.233 * frameTimeCounter))) * 43758.5453) - 0.5) * GRAIN_STRENGTH;
    color.rgb += vec3(noise);
    #endif

    #if COLOR_DIST == 1
    float colorDistort = COLOR_DIST_STRENGTH * sin(frameTimeCounter * 2.0);
    color.rgb *= vec3(1.0 + colorDistort, 1.0 - colorDistort, 1.0 + colorDistort);
    #endif


	// NVG
	#if ENABLE_NVG_IsSneaking == 1
	if(isSneaking == 1.0)
	{
		float gray = dot(color.rgb, vec3(0.299, 0.587, 0.114));
		vec3 grayscale = vec3(gray);
		vec3 colorTransform = vec3(R, G, B) * degree_brightness_increase;
		color = grayscale * colorTransform;
	}
	
	#else
	#if NVG == 1
    float gray = dot(color.rgb, vec3(0.299, 0.587, 0.114));
    vec3 grayscale = vec3(gray);
    vec3 colorTransform = vec3(R, G, B) * degree_brightness_increase;
    color = grayscale * colorTransform;
	#endif
	
	#endif

    // Apply saturation, brightness, and contrast
    float average = (color.r + color.g + color.b) / 3.0;
    color.rgb = mix(vec3(average), color.rgb, SATURATION);
    color.rgb += BRIGHTNESS;
    color.rgb = ((color.rgb - 0.5) * CONTRAST) + 0.5;
	
    #if VIGNETTE == 1
    float vignette = smoothstep(VIGNETTE_RADIUS, VIGNETTE_RADIUS + VIGNETTE_STRENGTH, distance);
    color = mix(color, vec3(0.0), vignette);
    #endif

    #if BLACK_STRIPES == 1
    float leftStripe = smoothstep(BLACK_STRIPES_WIDTH, BLACK_STRIPES_WIDTH - BLACK_STRIPES_SOFT, TexCoords.x);
    float rightStripe = smoothstep(1.0 - BLACK_STRIPES_WIDTH, 1.0 - (BLACK_STRIPES_WIDTH - BLACK_STRIPES_SOFT), TexCoords.x);
    float stripeEffect = max(leftStripe, rightStripe);
    color = mix(color, vec3(0.0), stripeEffect);
	#endif

    gl_FragColor = vec4(color, 1.0);
}
